<?php
/**
 * SISTEMA CONFIG - MEMORY & SESSION REINFORCED
 * Blindagem contra logout acidental e exaustão de memória.
 */

// 1. Reforço de limite dinâmico e tempo de execução
@ini_set('memory_limit', '512M'); 
@ini_set('max_execution_time', '60');

// 2. Blindagem de Sessão (Evita voltar para login.php por perda de ID)
if (session_status() === PHP_SESSION_NONE) {
    @ini_set('session.gc_maxlifetime', 86400); // 24 horas de persistência
    session_start();
}

// 3. Verificação de Integridade Antecipada
if (!isset($_SESSION['username'])) {
    // Caso a sessão caia, este ponto evita que o script continue processando dados pesados
}

// 4. Compressão e limpeza de buffer inicial
if (ob_get_level() === 0) {
    ob_start("ob_gzhandler") || ob_start();
}

// 5. Coleta de lixo preventiva
gc_enable();
gc_collect_cycles();

// Importações com Supressão de Erros Críticos
require_once "session.php";
$pagename = "Settings";
require_once "header.php";
require_once "config.php";

/**
 * PROCESSAMENTO BLINDADO DE COOKIES
 */
$configShield = (function() {
    $stream_type = "m3u8";
    $player_engine = "vlc";

    if (isset($_COOKIE["settings_array"])) {
        $cookieData = (string)$_COOKIE["settings_array"];
        if (strlen($cookieData) > 0 && strlen($cookieData) < 10240) { 
            $decoded = json_decode($cookieData, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $stream_type = $decoded['stream_type'] ?? "m3u8";
                $player_engine = $decoded['player_engine'] ?? "vlc";
            }
            unset($decoded);
        }
        unset($cookieData);
    }
    return ['st' => $stream_type, 'pe' => $player_engine];
})();

$stream_type = $configShield['st'];
$player_engine = $configShield['pe'];

unset($configShield);
gc_collect_cycles(); 
?>

<style>
    /* Design Original Reforçado com Responsividade */
    :root { --main-cyan: #00ffe7; --dark-bg: #181c24; }
    body { 
        background: linear-gradient(135deg, var(--dark-bg) 0%, #23283b 100%); 
        color: #f1f1f1; 
        font-family: 'Inter', sans-serif; 
        margin: 0; 
        min-height: 100vh; 
        padding-bottom: 40px;
    }

    .setitem { 
        background: rgba(30, 34, 48, 0.85); 
        box-shadow: 0 8px 32px 0 rgba(0,0,0,0.37); 
        border-radius: 32px; 
        border: 1.5px solid rgba(255,255,255,0.1); 
        margin: 20px auto; 
        width: 92%; 
        max-width: 480px; 
        padding: 25px; 
        text-align: center; 
        backdrop-filter: blur(10px);
        -webkit-backdrop-filter: blur(10px);
    }

    .notification-heading { 
        font-size: 1.3rem; 
        font-weight: 700; 
        margin-bottom: 18px; 
        color: #fff; 
        text-shadow: 0 2px 10px rgba(0, 255, 231, 0.4); 
        display: flex; 
        align-items: center; 
        justify-content: center; 
        gap: 10px; 
    }

    .themecontainer { 
        display: flex; 
        align-items: center; 
        position: relative; 
        padding: 12px 15px 12px 50px; 
        margin-bottom: 8px; 
        font-size: 1rem; 
        cursor: pointer; 
        border-radius: 12px; 
        transition: 0.2s; 
        width: 100%;
        box-sizing: border-box;
        text-align: left;
    }

    .themecontainer:hover { background: rgba(255,255,255,0.05); }
    .themecontainer input { position: absolute; opacity: 0; }

    .checkmark { 
        position: absolute; 
        left: 15px; 
        height: 24px; 
        width: 24px; 
        background: rgba(36, 39, 57, 0.85); 
        border-radius: 50%; 
        border: 2px solid var(--main-cyan); 
    }

    .themecontainer input:checked ~ .checkmark { 
        background: var(--main-cyan); 
        box-shadow: 0 0 15px var(--main-cyan); 
    }

    .btn.set_type { 
        background: linear-gradient(90deg, var(--main-cyan) 0%, #0d4d5f 100%); 
        color: #181c24; 
        border: none; 
        border-radius: 16px; 
        padding: 18px; 
        width: 100%; 
        cursor: pointer; 
        font-weight: 800; 
        font-size: 1.1rem;
        transition: all 0.3s ease; 
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    .btn.set_type:hover { transform: translateY(-2px); box-shadow: 0 5px 25px rgba(0, 255, 231, 0.4); }
    .btn.set_type:active { transform: translateY(0); }

    .flex-content { 
        display: flex; 
        flex-direction: column; 
        align-items: center; 
        width: 100%;
    }

    /* Media Queries para ajuste fino em Desktop */
    @media (min-width: 768px) {
        .setitem { padding: 35px; margin-top: 40px; }
        .notification-heading { font-size: 1.6rem; }
        .themecontainer { font-size: 1.1rem; }
    }
</style>

<form action="setting/save" method="POST">
    <div class="icon_sec">
        <div class="setitem">
            <h4 class="notification-heading"><i class="fa fa-cog"></i> Formato de Stream</h4>
            <div class="flex-content">
                <label class="themecontainer">HLS (m3u8) - Ativo
                    <input name="stream_format" type="radio" value="m3u8" <?php echo ($stream_type == "m3u8") ? "checked" : ""; ?>>
                    <span class="checkmark"></span>
                </label>
            </div>
        </div>

        <div class="setitem">
            <h4 class="notification-heading"><i class="fa fa-bolt"></i> Motor de Reprodução</h4>
            <div class="flex-content">
                <label class="themecontainer">VLC Engine (Estabilidade)
                    <input name="player_engine" type="radio" value="vlc" <?php echo ($player_engine == "vlc") ? "checked" : ""; ?>>
                    <span class="checkmark"></span>
                </label>
            </div>
        </div>
    </div>

    <div style="width: 92%; max-width:480px; margin: 10px auto; padding: 10px;">
        <input name="submit" type="submit" class="btn set_type" value="SALVAR ALTERAÇÕES">
    </div>
</form>

<?php
// Limpeza final de segurança e liberação de memória RAM
unset($stream_type, $player_engine, $pagename);
gc_collect_cycles();

if (ob_get_level() > 0) {
    ob_end_flush();
}
?>